/*
 * Monitor all the switches in the fabric.  We fire off a thread for
 * each switch which will monitor it periodically and update status
 * in memory for the main thread to look at.
 *
 * This thread will also look for changes in the switches (new or removed
 * linecards) and report that as needed.
 */
#include <pthread.h>
#include <unistd.h>
#include <time.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_switch.h"
#include "lf_scheduler.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_switch.h"
#include "fms_switch_monitor.h"
#include "fms_notify.h"
#include "fms_resolve.h"
#include "fms_state.h"
#include "fms_settings.h"

/*
 * local prototypes
 */
static void *fms_switch_query_thread(void *v);
static void fms_switch_queries_done(void *v);
static void fms_switch_inspect(struct lf_enclosure *ep);
static void fms_xbar_inspect(struct lf_xbar *xbp, int baseline);
static void fms_xcvr_inspect(struct lf_xcvr *xcp, int baseline);
static void fms_handle_switch_changes(struct lf_enclosure *ep);
static void fms_switch_lf_monitor(void *v);
static void fms_switch_do_lf_enclosure(struct lf_enclosure *ep);
static void fms_switch_vlf_monitor(void *v);
static void fms_switch_do_vlf_enclosure(struct lf_enclosure *ep);
static void fms_get_enclosure_types(struct lf_fabric *fp);
static void fms_switch_start_queries(void *v);

/*
 * Initialize variables for switch monitoring
 */
int
init_switch_vars()
{
  LF_CALLOC(F.switch_vars, struct fms_switch, 1);
  return 0;

 except:
  return -1;
}

/*
 * start all the switch monitoring threads
 */
int
init_switch_monitoring()
{
  struct lf_fabric *fp;
  struct fms_settings *fsetp;
  struct fms_switch *fswp;
  struct lf_event *event;
  int rc;
  int e;

  fp = F.fabvars->fabric;
  fsetp = F.settings;
  fswp = F.switch_vars;

  /* set up the pthread stuff */
  pthread_cond_init(&fswp->switch_cond, NULL);
  pthread_mutex_init(&fswp->switch_mutex, NULL);
  fswp->num_switch_monitors = 0;

  /* Make sure all enclosures have a type */
  fms_get_enclosure_types(fp);

  /* start the query thread for each enclosure */
  pthread_mutex_lock(&fswp->switch_mutex);
  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;

    ep = fp->enclosures[e];

    /* don't try monitoring generic (bogus) switches */
    if (strcmp(ep->product_id, "GENSW") != 0) {

      rc = fms_monitor_enclosure(ep);
      if (rc == -1) LF_ERROR(("Error initiating enclosure monitoring"));


      ++fswp->num_switch_monitors;
    }
  }

  /* set the number of pending queries, and enable them to finish */
  fswp->switch_queries_pending = fswp->num_switch_monitors;
  pthread_mutex_unlock(&fswp->switch_mutex);

  /* Start the low-frequency switch inspection process */
  event = lf_schedule_event(fms_switch_lf_monitor, NULL,
                            fsetp->low_freq_monitor_interval*1000);
  if (event == NULL) LF_ERROR(("Error scheduling event"));

  /* Start the very-low-frequency switch inspection process */
  event = lf_schedule_event(fms_switch_vlf_monitor, NULL,
                            fsetp->very_low_freq_monitor_interval*1000);
  if (event == NULL) LF_ERROR(("Error scheduling event"));
  return 0;

 except:
  return -1;
}

/*
 * Thread to actually perform the monitoring
 */
static void *
fms_switch_query_thread(
  void *v)
{
  struct lf_enclosure *ep;
  struct fms_switch *fswp;
  int rc;
  int ok;

  /* initialize thread-local error handler */
  fms_init_error();

  fswp = F.switch_vars;
  ep = (struct lf_enclosure *)v;	/* switch we should monitor */
  ok = TRUE;

  while (1) {
    fms_notify(FMS_EVENT_DEBUG, "querying %s", ep->name);

    rc = lf_query_switch(ep->name, &ep);
    if (rc == -1) {
      if (ok) {
	fms_perror();
	fms_switch_alert_cannot_read_enclosure(ep);
	ok = FALSE;
      }
      fms_notify(FMS_EVENT_DEBUG, "Error querying switch %s", ep->name);

    /* switch read OK */
    } else {

      /* If switch was unreadable before, rescind alert about it now */
      if (!ok) {
	fms_switch_alert_enclosure_read_ok(ep);
	ok = TRUE;
      }

      /* If any changes, save them away to be dealt with */
      if (ep->data->change_list != NULL) {
	if (FMS_ENC(ep)->change_list == NULL) {
	  FMS_ENC(ep)->change_list = ep->data->change_list;
	  ep->data->change_list = NULL;

	/* If changes pending, ignore these */
	} else {
	  struct lf_switch_data_change *cp;
	  struct lf_switch_data_change *ncp;

	  /* Clear out the list */
	  cp = ep->data->change_list;
	  ep->data->change_list = NULL;

	  /* Free all the change records */
	  while (cp != NULL) {
	    ncp = cp->next;
	    lf_free_switch_change(cp);
	    cp = ncp;
	  }
	}
      }

      fms_notify(FMS_EVENT_DEBUG, "done querying %s", ep->name);
    }

    /* Decrement pending count.  When it reaches 0, schedule the
     * "query_done" task in the main thread
     */
    pthread_mutex_lock(&fswp->switch_mutex);
    --fswp->switch_queries_pending;

    /* if nothing else pending schedule the inspect task */
    if (fswp->switch_queries_pending <= 0) {
      fswp->switch_query_done_task =
	  lf_schedule_event(fms_switch_queries_done, NULL, 0);
      if (fswp->switch_query_done_task == NULL) {
	pthread_mutex_unlock(&fswp->switch_mutex);
	LF_ERROR(("Scheduling switch inspection event"));
      }
    }

    /* Now, wait for the main thread to cue us to run again */
    pthread_cond_wait(&fswp->switch_cond, &fswp->switch_mutex);
    pthread_mutex_unlock(&fswp->switch_mutex);
  }

 except:
  /* XXX inidicate to caller he should join on us to cleanup? schedule an
   * event to run the join in 5 seconds or so */
  fms_perror();
  return NULL;
}

/*
 * Start monitoring an enclosure
 */
int
fms_monitor_enclosure(
  struct lf_enclosure *ep)
{
  int rc;

  /* make sure this enclosure has a monitor structure */
  if (FMS_ENC(ep)->monitor == NULL) {
    LF_CALLOC(FMS_ENC(ep)->monitor, struct fms_switch_monitor, 1);
  }

  /*
   * start a monitoring thread on this enclosure
   */
  rc = pthread_create(&(FMS_ENC(ep)->monitor->tid),
      NULL, fms_switch_query_thread, ep);
  if (rc != 0) {
    LF_ERROR(("Error starting query thread for %s, increase stack limit?",
	  ep->name));
  }
  return 0;

 except:
  return -1;
}

/*
 * Scan through all enclosures and make sure we have the type for
 * each one.  This will also create all needed backplane slots
 * as a side-effect.
 */
static void
fms_get_enclosure_types(
  struct lf_fabric *fp)
{
  int e;
  int rc;

  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;

    ep = fp->enclosures[e];

    /* if product ID is NULL or empty, query the switch now */
    if (ep->product_id == NULL || ep->product_id[0] == '\0') {
      int s;

      fms_notify(FMS_EVENT_INFO, "Initial query of %s", ep->name);

      /* query the switch to learn its type */
      rc = lf_query_switch(ep->name, &ep);
      if (rc == -1) {
	LF_ERROR(("Error getting type for %s\n", ep->name));
      }

      /* make sure we have a product ID now */
      if (ep->product_id == NULL || ep->product_id[0] == '\0') {
	LF_ERROR(("Failed to get switch type for %s\n", ep->name));
      }

      fms_notify(FMS_EVENT_INFO, "Switch %s type is %s",
	ep->name, ep->product_id);

      /* Make sure all linecards have FMS-private data */
      for (s=0; s<ep->num_slots; ++s) {
	struct lf_linecard *lp;

	lp = ep->slots[s];
	if (lp != NULL) {
	  fms_fill_linecard_private(lp);
	}
      }

      /*
       * If any changes, move them to the work list and deal with them now
       */
      if (ep->data->change_list != NULL) {
	if (FMS_ENC(ep)->change_list == NULL) {
	  FMS_ENC(ep)->change_list = ep->data->change_list;
	  ep->data->change_list = NULL;

	  fms_handle_switch_changes(ep);
	}
      }

      /* Update the database to contain proper enclosure info */
      fms_remove_enclosure_from_db(ep);
      fms_add_enclosure_to_db(ep);
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Handle any changes detected from a switch enclosure
 */
static void
fms_handle_switch_changes(
  struct lf_enclosure *ep)
{
  struct lf_switch_data_change *cp;
  struct lf_switch_data_change *ncp;

  /* Process each change in the list */
  cp = FMS_ENC(ep)->change_list;

  /* If nothing to do, just return */
  if (cp == NULL) return;

  FMS_ENC(ep)->change_list = NULL;

  /* process each change in the list */
  while (cp != NULL) {
    ncp = cp->next;	/* save the next pointer for later */

    switch (cp->dc_type) {
      struct lf_linecard *lp;
      int slot;

      case LF_SWITCH_CHANGE_MISSING_LINECARD:
	slot = cp->c.missing_linecard.slot;
	fms_notify(FMS_EVENT_INFO, "%s, slot %d (%s) is missing",
	    ep->name, slot, ep->slots[slot]->product_id);
	/* XXX deal with missing line card */
	break;

      case LF_SWITCH_CHANGE_NEW_LINECARD:
	slot = cp->c.new_linecard.slot;

	fms_notify(FMS_EVENT_INFO, "%s, new card at slot %d (%s)", 
	    ep->name, slot, cp->c.new_linecard.product_id);

	lp = fms_new_linecard(ep, slot, cp->c.new_linecard.product_id,
				  cp->c.new_linecard.serial_no);

	/* Check to see if we can replace any bogus enclosures with this
	 * linecard
	 */
	fms_merge_new_linecard(lp);

	break;

      case LF_SWITCH_CHANGE_CHANGED_LINECARD:
	slot = cp->c.new_linecard.slot;
	lp = ep->slots[slot];

	fms_notify(FMS_EVENT_INFO, "%s, slot %d changed to %s",
	    ep->name, lf_slot_display_no(lp), cp->c.new_linecard.product_id);

	if (lp == NULL) LF_ERROR(("Unexpected null lp"));

printf("old ID=%s, new ID=%s\n", lp->product_id, cp->c.new_linecard.product_id);

	/* If product ID is different, XXX */
	if (strcmp(cp->c.new_linecard.product_id, lp->product_id) != 0) {
	  LF_ERROR(("XXX changing product_id not supported yet"));
	}

	/* If serial number is different, update and update any xbars IDs */
	if (strcmp(cp->c.new_linecard.serial_no, lp->serial_no) != 0) {
	  int x;

	  /* copy over the new serial number */
	  LF_DUP_STRING(lp->serial_no, cp->c.new_linecard.serial_no);

	  /* update xbar ID for each xbar */
	  for (x=0; x<lp->num_xbars; ++x) {
	    struct lf_xbar *xp;

	    xp = LF_XBAR(lp->xbars[x]);
	    xp->xbar_id = lf_get_xbar_id(lp, x);
	  }
	}
	break;
    }

    /* Free each record after processing */
    lf_free_switch_change(cp);
    cp = ncp;
  }

  /* stuff may have changed */
  fms_state_fabric_changed(FMS_STATE_FABRIC_CHANGE);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * This is scheduled to run each time a set of switch queries completes.
 */
static void
fms_switch_queries_done(
  void *v)
{
  int e;
  struct lf_enclosure *ep;
  struct fms_switch *fswp;
  struct lf_fabric *fp;

  fswp = F.switch_vars;
  fp = F.fabvars->fabric;

  fms_notify(FMS_EVENT_DEBUG, "All switch queries done, processing");

  /* Make sure timer is NULLed */
  fswp->switch_query_done_task = NULL;

  /* get pointer to enclosure we've been called on */
  for (e=0; e<fp->num_enclosures; ++e) {
    ep = fp->enclosures[e];

    /* don't inspect generic (bogus) switches */
    if (strcmp(ep->product_id, "GENSW") != 0) {

      /* perform the switch inspection */
      fms_switch_inspect(ep);
    }
  }

  /* If there is a deferred handler, set that up to be called next time */
  if (fswp->res_deferred_handler != NULL) {
    fswp->res_handler = fswp->res_deferred_handler;
    fswp->res_deferred_handler = NULL;

    /* since we had a deferred handler, start queries now */
    fms_switch_start_queries(NULL);

  /* No deferred handler, schedule next set of queries */
  } else {
    fswp->switch_start_query_timer = lf_schedule_event(
	fms_switch_start_queries, NULL,
	F.settings->switch_query_interval * 1000);
    if (fswp->switch_start_query_timer == NULL) {
      LF_ERROR(("Error scheduling switch query timer"));
    }
  }

  /*
   * All inspections done, call fabric resolution handler if there is one
   * This needs to be done *after* start_query_timer is initiated because the
   * handler may call set_handler which will then cancel the timer.  If he 
   * tries to cancel it, and then we schedule it again, chaos will ensue.
   */
  if (fswp->res_handler != NULL) {
    void (*handler)(void);

    /* save the handler, NULL it, then call it since handler may set 
     * a new handler and set_handler wants the handler to be NULL for safety
     */
    handler = fswp->res_handler;
    fswp->res_handler = NULL;
    (*handler)();
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Start the next set of switch queries.
 */
static void
fms_switch_start_queries(
  void *v)
{
  struct fms_switch *fswp;

  fswp = F.switch_vars;

  /* ensure that all threads are waiting on condition */
  pthread_mutex_lock(&fswp->switch_mutex);
  fswp->switch_queries_pending = fswp->num_switch_monitors;
  pthread_mutex_unlock(&fswp->switch_mutex);

  /* Release the hounds! */
  fms_notify(FMS_EVENT_DEBUG, "Starting next round of switch queries");
  pthread_cond_broadcast(&fswp->switch_cond);
}

/*
 * Set a handler for the fabric resolution stuff.
 * This will get made more generic if anyone else needs handlers, but
 * not right now.  So there.
 *
 * If queries are currently pending, save this as a "deferred handler" and
 * we will plug it in when this set of queries is complete, and then immediately
 * restart the query process.
 *
 * If no queries are currently pending, then we will set the handler
 * and reschedule the query start routine to begin immediately.
 */
void
fms_switch_set_res_handler(
  void (*handler)(void))
{
  struct fms_switch *fswp;

  fswp = F.switch_vars;

  /* Make sure both are NULL right now. */
  if (fswp->res_handler != NULL) {
    LF_ERROR(("res_handler must be NULL!"));
  }
  if (fswp->res_deferred_handler != NULL) {
    LF_ERROR(("res_deferred_handler must be NULL!"));
  }

  /* If queries pending, save this as deferred handler */
  if (fswp->switch_queries_pending > 0) {
    fswp->res_deferred_handler = handler;

  /* nothing pending, save the handler and start queries now */
  } else {

    /* save the handler */
    fswp->res_handler = handler;

    /* It is possible queries have completed but not yet been serviced.
     * In this case, just let the service routine run, which it will do
     * immediately.  Otherwise, expect
     * the start timer to be set, clear it, and call start routine directly.
     */
    if (fswp->switch_query_done_task == NULL) {
      if (fswp->switch_start_query_timer == NULL) {
	LF_ERROR(("Error: switch_start_query_timer should be set!"));
      }

      /* un-schedule the call to fms_switch_start_queries() and call it */
      lf_remove_event(fswp->switch_start_query_timer);
      fswp->switch_start_query_timer = NULL;
      fms_switch_start_queries(NULL);
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Clear fabric resolution handler
 */
void
fms_switch_clear_res_handler()
{
  struct fms_switch *fswp;

  fswp = F.switch_vars;

  /* clear handler and deferred handler */
  fswp->res_handler = NULL;
  fswp->res_deferred_handler = NULL;
}

/*
 * Inspect a switch for changes we might want to report
 */
static void
fms_switch_inspect(
  struct lf_enclosure *ep)
{
  struct fms_settings *fsetp;
  int baseline;
  int l;

  fsetp = F.settings;

  /* inspect all the linecards */
  for (l=0; l<ep->num_slots; ++l) {
    struct lf_linecard *lp;
    struct lf_linecard_data *ldp;
    int xb;
    int xc;

    lp = ep->slots[l];
    if (lp == NULL) continue;

    /* If this card is new, just mark it not new and skip it.  This will allow
     * us to create a baseline
     */
    if (FMS_LC(lp)->new) {
      FMS_LC(lp)->new = FALSE;
      baseline = TRUE;
    } else {
      baseline = FALSE;
    }

    ldp = lp->data;

    /* inspect all xbars */
    for (xb=0; xb<lp->num_xbars; ++xb) {
      fms_xbar_inspect(LF_XBAR(lp->xbars[xb]), baseline);
    }

    /* inspect all xcvrs */
    for (xc=0; xc<lp->num_xcvrs; ++xc) {
      fms_xcvr_inspect(LF_XCVR(lp->xcvrs[xc]), baseline);
    }

    /* Skip to the next card if just getting baseline */
    if (baseline) continue;

    /* If newly hot, raise an alert.  If not hot, rescind alert if
     * below hysteresis amount
     */
    if (!FMS_LC(lp)->hot
	&& (ldp->cur_vals.temperature[0] > fsetp->linecard_overtemp_threshold_0
	    || ldp->cur_vals.temperature[1]
	       > fsetp->linecard_overtemp_threshold_1)) {
      FMS_LC(lp)->hot = TRUE;
      fms_switch_alert_linecard_hot(lp);

    } else if (FMS_LC(lp)->hot
	       && ldp->cur_vals.temperature[0]
	          < (fsetp->linecard_overtemp_threshold_0
		     - fsetp->linecard_overtemp_hysteresis_0)
	       && ldp->cur_vals.temperature[1]
	          < (fsetp->linecard_overtemp_threshold_1
		     - fsetp->linecard_overtemp_hysteresis_1)) {
      FMS_LC(lp)->hot = FALSE;
      fms_switch_alert_linecard_cool(lp);
    }

    /* If overtemps have occurred, raise an alert */
    if (ldp->cur_vals.overtempcount > ldp->old_vals.overtempcount) {
      fms_switch_alert_linecard_overtemp(lp);
    }
  }

  /* handle changes, if any */
  fms_handle_switch_changes(ep);
}

/*
 * Inspect all the ports on an xbar and report anything reportable
 */
void
fms_xbar_inspect(
  struct lf_xbar *xp,
  int baseline)
{
  int p;
  struct lf_xbarport_data *dp;
  int delta;

  /*
   * See if quadrant disable changed.
   * This is done independent of baseline or not.
   */
  if (xp->xbar_data->cur_vals.quadrant_disable != xp->quadrant_disable) {

    fms_notify(FMS_EVENT_DEBUG, "%s, slot %d, x%d qd = %d (was %d)",
	xp->linecard->enclosure->name, lf_slot_display_no(xp->linecard),
	xp->xbar_no, xp->xbar_data->cur_vals.quadrant_disable,
	xp->quadrant_disable);
    
    xp->quadrant_disable = xp->xbar_data->cur_vals.quadrant_disable;
    fms_state_fabric_changed(FMS_STATE_QUAD_DIS_CHANGE);
  }

  /* all done if just creating baseline */
  if (baseline) return;

  /* check out every port */
  for (p=0; p<xp->num_ports; ++p) {
    struct fms_xbar_port *fxpp;

    dp = xp->data + p;		/* data for this port */
    fxpp = FMS_XBAR(xp)->x_port_data[p];

    /*
     * update some counters for the low-frequency check
     */
    delta = dp->cur_vals.badcrcs - dp->old_vals.badcrcs;
    if (delta > 0) {
      fxpp->lfmon.badcrcs += delta;
    }
    delta = dp->cur_vals.portflipcount - dp->old_vals.portflipcount;
    if (delta > 0) {
      fxpp->vlfmon.portflipcount += delta;
    }

    /* newly disabled? */
    if (dp->cur_vals.control != 0 && dp->old_vals.control == 0) {
      fms_switch_alert_xbarport_disabled(xp, p);
    }

    /* newly re-enabled? */
    if (dp->cur_vals.control == 0 && dp->old_vals.control != 0) {
      fms_switch_alert_xbarport_enabled(xp, p);
    }

    /* newly down? */
    if (dp->cur_vals.portdown != 0 && dp->old_vals.portdown == 0) {

      /* this matters only if something is attached here */
      if (xp->topo_ports[p] != NULL) {
	int changed;


	/* mark the link down in the fabric */
	changed = fms_xbar_link_down(xp, p);

	/* If state changed, raise an alert */
	if (changed) {
	  fms_switch_alert_xbar_link_down(xp, p);	/* raise the alert */
	}
      }
    }

    /* newly up? */
    if (dp->cur_vals.portdown == 0 && dp->old_vals.portdown != 0) {
      fms_switch_alert_xbar_link_up(xp, p);	/* relic any alerts */

      /* mark it as available */
      fms_xbar_link_up(xp, p);	/* mark the link up in the fabric */
    }
  }
}

/*
 * Inspect a transceiver's data and report anything reportable
 */
void
fms_xcvr_inspect(
  struct lf_xcvr *xcp,
  int baseline)
{
  struct lf_xcvr_data *dp;

  /* nothing to do 1st time through */
  if (baseline) return;

  dp = xcp->data;		/* get pointer to data struct */

  /* newly disabled? */
  if (dp->cur_vals.control != 0 && dp->old_vals.control == 0) {
    fms_switch_alert_xcvr_disabled(xcp);
  }

  /* newly re-enabled? */
  if (dp->cur_vals.control == 0 && dp->old_vals.control != 0) {
    fms_switch_alert_xcvr_enabled(xcp);
  }

#if 0		/* don't know if we really want this or not */

  /* signal lost? report only if there is something attached */
  if (dp->cur_vals.signallost != 0 && dp->old_vals.signallost == 0) {
    int p;
    for (p=0; p<xcp->num_conns; ++p) {
      if (xcp->ports[p] != NULL) {
	fms_switch_alert_xcvr_signal_lost(xcp);
	fms_switch_xcvr_disconnected(xcp);
	break;
      }
    }
  }

  /* signal found? */
  if (dp->cur_vals.signallost == 0 && dp->old_vals.signallost != 0) {
    fms_switch_alert_xcvr_signal_ok(xcp);
  }
#endif
}

/*
 * Low frequency inspection for an enclosure
 */
static void
fms_switch_do_lf_enclosure(
  struct lf_enclosure *ep)
{
  struct fms_settings *fsetp;
  int l;

  fsetp = F.settings;

  for (l=0; l<ep->num_slots; ++l) {
    struct lf_linecard *lp;
    int x;

    lp = ep->slots[l];
    if (lp == NULL) continue;

    /* process the xbars on this linecard */
    for (x=0; x<lp->num_xbars; ++x) {
      struct lf_xbar *xp;
      int p;

      xp = LF_XBAR(lp->xbars[x]);

      /* process the ports on this xbar */
      for (p=0; p<xp->num_ports; ++p) {
      	struct fms_xbar_port *fxpp;

	fxpp = FMS_XBAR(xp)->x_port_data[p];	/* per-port data */

	/* check for fatal badcrc count */
	if (fxpp->lfmon.badcrcs > fsetp->lf_fatal_badcrc_threshold) {
	  fms_switch_alert_badcrc_count(xp, p, /* TRUE */ FALSE);
	
	/* check for non-fatal badcrc count */
	} else if (fxpp->lfmon.badcrcs > fsetp->lf_badcrc_threshold) {
	  fms_switch_alert_badcrc_count(xp, p, FALSE);
	}

	/* clear all low-frequency counters */
	memset(&fxpp->lfmon, 0, sizeof(fxpp->lfmon));
      }
    }
  }
}

/*
 * Low-frequency switch monitor
 */
static void
fms_switch_lf_monitor(
  void *v)
{
  struct lf_event *event;
  struct lf_fabric *fp;
  struct fms_settings *fsetp;
  int e;

  fp = F.fabvars->fabric;
  fsetp = F.settings;

  fms_notify(FMS_EVENT_DEBUG, "In low-frequency switch monitor");

  /* process each enclosure */
  for (e=0; e<fp->num_enclosures; ++e) {
    fms_switch_do_lf_enclosure(fp->enclosures[e]);
  }

  /* re-start the low-frequency switch inspection process */
  event = lf_schedule_event(fms_switch_lf_monitor, NULL,
                            fsetp->low_freq_monitor_interval*1000);
  if (event == NULL) LF_ERROR(("Error rescheduling event"));
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Very low frequency inspection for an enclosure
 */
static void
fms_switch_do_vlf_enclosure(
  struct lf_enclosure *ep)
{
  struct fms_settings *fsetp;
  int l;

  fsetp = F.settings;

  for (l=0; l<ep->num_slots; ++l) {
    struct lf_linecard *lp;
    int x;

    lp = ep->slots[l];
    if (lp == NULL) continue;

    /* process the xbars on this linecard */
    for (x=0; x<lp->num_xbars; ++x) {
      struct lf_xbar *xp;
      int p;

      xp = LF_XBAR(lp->xbars[x]);

      /* process the ports on this xbar */
      for (p=0; p<xp->num_ports; ++p) {
      	struct fms_xbar_port *fxpp;

	fxpp = FMS_XBAR(xp)->x_port_data[p];	/* per-port data */

	/* check port flip count */
	if (fxpp->vlfmon.portflipcount > fsetp->vlf_portflip_threshold) {
	  fms_switch_alert_updown_count(xp, p);
	}

	/* clear all low-frequency counters */
	memset(&fxpp->vlfmon, 0, sizeof(fxpp->vlfmon));
      }
    }
  }
}

/*
 * Very low-frequency switch monitor
 */
static void
fms_switch_vlf_monitor(
  void *v)
{
  struct lf_event *event;
  struct lf_fabric *fp;
  struct fms_settings *fsetp;
  int e;

  fp = F.fabvars->fabric;
  fsetp = F.settings;

  fms_notify(FMS_EVENT_DEBUG, "In very-low-frequency switch monitor");

  /* process each enclosure */
  for (e=0; e<fp->num_enclosures; ++e) {
    fms_switch_do_vlf_enclosure(fp->enclosures[e]);
  }

  /* re-start the very-low-frequency switch inspection process */
  event = lf_schedule_event(fms_switch_vlf_monitor, NULL,
                            fsetp->very_low_freq_monitor_interval*1000);
  if (event == NULL) LF_ERROR(("Error rescheduling event"));
  return;

 except:
  fms_perror_exit(1);
}
